implementation module deltaBitmap


//	Clean 0.8 I/O library.
//	Interface functions for drawing bitmaps.


import StdArray, StdBool, StdFile, StdInt, StdClass
import ioTypes, picture


BitmapError :: String String -> .x
BitmapError rule error
	= Error rule "Bitmap" error

::	Bitmap
	=	{	originalWidth	:: !Int,			// The width of the bitmap
			originalHeight	:: !Int,			// The height of the bitmap
			resizedWidth	:: !Int,
			resizedHeight	:: !Int,
			bitmapContents	:: !{#Char}		// The bitmap information
		}

openBitmap :: !{#Char} !*env -> (!(!Bool,!Bitmap),!*env)	| FileSystem env
openBitmap name env
	# (ok,file,env)		= fopen name FReadData env
	| not ok
	= ((ok,noBitmap),env)
	# (ok,bitmap,file)	= readBitmap file
    # (_,env)			= fclose file env
	= ((ok,bitmap),env)
where
	noBitmap			= { originalWidth=0,originalHeight=0,
				 			resizedWidth=0,  resizedHeight=0,
							bitmapContents={} 
						  }
	
	// readBitmap reads a bitmap from a file. See page 176 of Programming Windows 95 (Charles Petzold)
	readBitmap :: !*File -> (!Bool,!Bitmap,!*File)
	readBitmap file
		# (_, c1,file) = freadc file
		# (ok,c2,file) = freadc file      // read first two bytes
		| not ok || c1<>'B' || c2<>'M'	  // are they "BM"? 
		= (False,noBitmap,file)
		# (_,  fileSize,file)	= freadi file // read file size
		# (_,  _,       file)	= freadi file // skip bfReserved1 & 2
		# (_,  _,       file)	= freadi file // skip bfOffBits
		# (_,  _,       file)	= freadi file // skip biSize
		# (_,  w,       file)	= freadi file // read width
		# (ok1,h,       file)	= freadi file // read height
		# (ok2,         file)	= fseek  file 0 FSeekSet
		| not ok1 || not ok2
		= (False,noBitmap,file)
		# (data,file)			= freads file fileSize
		| size data <> fileSize
		= (False,noBitmap,file)
		| otherwise 
		= (	True,
			{	originalWidth=w, originalHeight=h,
	 			resizedWidth=w,  resizedHeight=h,
				bitmapContents=data
			},
			file
		  ) 

getBitmapSize :: !Bitmap -> (!Int,!Int)
getBitmapSize { resizedWidth,resizedHeight }
	= (resizedWidth,resizedHeight)

resizeBitmap :: !Bitmap !Vector -> Bitmap
resizeBitmap { originalWidth, originalHeight, bitmapContents } (newW,newH)
	= { originalWidth=originalWidth, originalHeight=originalHeight,
		resizedWidth=newW, resizedHeight=newH,
		bitmapContents=bitmapContents
	  }
	  
drawBitmap :: !Point !Rectangle !Bitmap !Picture -> Picture
drawBitmap pos part bitmap=:{ originalWidth,originalHeight,resizedWidth,resizedHeight,
							  bitmapContents } picture
	| originalWidth==0 || originalHeight==0 || size bitmapContents==0
	= picture
	# (x1,y1,x2,y2)	= RectangleToRect part
	  (x1,y1,x2,y2)	= (	SetBetween x1 0 resizedWidth,SetBetween y1 0 resizedHeight,
	  					SetBetween x2 0 resizedWidth,SetBetween y2 0 resizedHeight
	  				  )
	| x1==x2 || y1==y2
	= picture
	# (hdc,tb)		= UnpackPicture picture
	| resizedWidth==originalWidth && resizedHeight==originalHeight
		// bitmap was not resized
		# (hdc,tb)		= WinDrawBitmap pos ((x1,y1),(x2,y2)) bitmapContents (hdc,tb)
		= PackPicture hdc tb
	// bitmap was resized
	# ox1 = (x1*originalWidth)/resizedWidth		// calculate linear projection to get rectangle of
	  oy1 = (y1*originalHeight)/resizedHeight	// original bitmap, which has to be drawn
	  ox2 = (x2*originalWidth)/resizedWidth
	  oy2 = (y2*originalHeight)/resizedHeight
 	  (hdc,tb)		= WinDrawResizedBitmap 	((ox1,oy1),(ox2,oy2)) pos (x2-x1) (y2-y1)
 	  										bitmapContents (hdc,tb)
	= PackPicture hdc tb

WinDrawBitmap :: !(!Int, !Int) !(!(!Int,!Int),!(!Int,!Int)) !{#Char} !(!Int,!*Int) -> (!Int,!*Int)
WinDrawBitmap _ _ _ _
  = code
	{
	.inline WinDrawBitmap
		ccall WinDrawBitmap "IIIIIISII-II"
	.end
	}

/*
drawResizedBitmap :: !Rectangle !Rectangle !Bitmap !Picture -> Picture
drawResizedBitmap src dest bitmap=:{ originalWidth,originalHeight,bitmapContents } picture
	| originalWidth==0 || originalHeight==0 || size bitmapContents==0
	= picture
	# (hdc,tb)		= UnpackPicture picture
	# (hdc,tb)		= WinDrawResizedBitmap src dest bitmapContents (hdc,tb)
	= PackPicture hdc tb
*/

WinDrawResizedBitmap :: !(!(!Int,!Int),!(!Int,!Int)) !(!Int,!Int) !Int !Int !{#Char} !(!Int,!*Int)
			 -> (!Int,!*Int)
WinDrawResizedBitmap _ _ _ _ _ _
  = code
	{
	.inline WinDrawResizedBitmap
		ccall WinDrawResizedBitmap "IIIIIIIISII-II"
	.end
	}
